/*
	Thingy Axon library

	BluetoothClient is part of Thingy library that handles communication to BLE module

	Copyright (c) 2017 Thingy.IO
*/

#include "BluetoothClient.h"


BluetoothClient::BluetoothClient(int baudRatei, long loopLengthi) : btSer(14, 13, BUFFER_SIZE)
{
	for (int i = 0; i < MAX_BEACONS; i++)
	{
		BeaconTable[i].lastActive = 0;
		BeaconTable[i].advertisementsCnt = 0;
	}
	loopLength = loopLengthi;
	btSer.begin(baudRatei);
}

bool BluetoothClient::setup(){
	//reset BT
	char temp[9] = "OK+DISC:";
	for (int i = 0; i < 8; i++)
	{
		BeaconResponse[i] = temp[i];
	}

	digitalWrite(4, LOW);
	delay(150);
	digitalWrite(4, HIGH);
	delay(1000);
	bool setupComplete = false;
	btSer.flush();
	if (isBTresponse("AT", "OK", 100)
			&& isBTresponse("AT+ROLE1", "OK+Set:1", 100)
			&& isBTresponse("AT+IMME1", "OK+Set:1", 100)
			)
	{
		setupComplete = true;
		Serial.println(F("Setup BT"));
	}
	else{
		Serial.println(F("Bluetooth Error"));
	}
	return setupComplete;
}

bool BluetoothClient::startBeacon(char* major, char* minor) //power control should be added
{
	char MajorCommand[14] = "AT+MARJ0xXXXX";
	char MinorCommand[14] = "AT+MINO0xXXXX";
	char MajorResponse[14] = "OK+Set:0xXXXX";
	char MinorResponse[14] = "OK+Set:0xXXXX";

	for (int i = 7; i < 13; i++)
	{
		MajorCommand[i] = major[i-7];
		MajorResponse[i] = major[i-7];

		MinorCommand[i] = minor[i-7];
		MinorResponse[i] = minor[i-7];
	}
	Serial.println(MajorCommand);
	Serial.println(MajorResponse);
	Serial.println(MinorCommand);
	Serial.println(MinorResponse);

	digitalWrite(4, LOW);
	delay(150);
	digitalWrite(4, HIGH);
	delay(1000);
	bool setupComplete = false;
	if (isBTresponse("AT", "OK", 100)
			&& isBTresponse("AT+ROLE0", "OK+Set:0", 100)
			&& isBTresponse("AT+IMME1", "OK+Set:1", 100)
			&& isBTresponse(MajorCommand, MajorResponse, 100)
			&& isBTresponse(MinorCommand, MinorResponse, 100)
			&& isBTresponse("AT+ADVI0", "OK+Set:0", 100)
			&& isBTresponse("AT+IBEA1", "OK+Set:1", 100)
			&& isBTresponse("AT+RESET", "OK+RESET", 100)
			)
	{
		setupComplete = true;
		Serial.println(F("Setup Beacon"));
	}
	else{
		Serial.println(F("Bluetooth Error"));
	}
	return setupComplete;
}

void BluetoothClient::readOneBeacon()
{
	char scannedBeacon[RESPONSE_LEN];
	readBytesBT(scannedBeacon, RESPONSE_LEN, 100);
	int lastIndex = 0;
	bool found = false;
	while((BeaconTable[lastIndex].lastActive != 0)&&(!found)&&(lastIndex < MAX_BEACONS))
	{
		if (!memcmp(scannedBeacon, BeaconTable[lastIndex].data, RESPONSE_LEN - 4)) //compare until RSSI
		{
			//Serial.print(" Found in table at ");
			//Serial.print(lastIndex);
			memcpy (BeaconTable[lastIndex].data, scannedBeacon, RESPONSE_LEN); //copy all... one can only copy RSSI
			BeaconTable[lastIndex].lastActive = millis();
			BeaconTable[lastIndex].advertisementsCnt++;
			found = true;
		}
		lastIndex++;
	}
	if (!found)
	{
		if (lastIndex < MAX_BEACONS) //new beacon, there is space
		{
			// Serial.print(" Put at the end ");
			memcpy (BeaconTable[lastIndex].data, scannedBeacon, RESPONSE_LEN);
			BeaconTable[lastIndex].lastActive = millis();
			BeaconTable[lastIndex].advertisementsCnt = 1;
		}
		else //new beacon, all full. replace the oldest one
		{
			//  Serial.print(" Replacing oldest ");
			long oldestTime = BeaconTable[0].lastActive;
			int oldestIndex = 0;
			for (int i = 1; i < MAX_BEACONS; i++)
			{
				if (BeaconTable[i].lastActive < oldestTime)
				{
					oldestTime = BeaconTable[i].lastActive;
					oldestIndex = i;
				}
			}
			memcpy (BeaconTable[oldestIndex].data, scannedBeacon, RESPONSE_LEN);
			BeaconTable[oldestIndex].lastActive = millis();
			BeaconTable[oldestIndex].advertisementsCnt = 1;
		}
	}
}

void BluetoothClient::deleteOldBeacons()
{
	int lastIndex = 0;
	while((BeaconTable[lastIndex].lastActive != 0&&(lastIndex < MAX_BEACONS))){
		lastIndex++;
	}
	lastIndex--;
	int i = lastIndex;

	while (i >= 0)
	{
		if((millis() - BeaconTable[i].lastActive) > DELETE_DEADLINE)
		{
			if (i != lastIndex){
				memcpy (BeaconTable[i].data, BeaconTable[lastIndex].data, RESPONSE_LEN);
				BeaconTable[i].lastActive = BeaconTable[lastIndex].lastActive;
				BeaconTable[i].advertisementsCnt = BeaconTable[lastIndex].advertisementsCnt;
			}

			//Serial.print(F(" Deleting oldest at "));
			//Serial.print(i);
			//Serial.println();
			BeaconTable[lastIndex].lastActive = 0;
			lastIndex--;
		}
		i--;
	}
}

void BluetoothClient::loop()
{
	//asynchronous read of the software serial. Return only beacons
	if (btSer.available())
	{
		system_soft_wdt_feed();
		btSer.setTimeout(100);
		if (btSer.find("OK+DISC:"))
		{
			readOneBeacon();
			deleteOldBeacons();
  		}
	}
}

bool BluetoothClient::readCommand(){
	if ((millis() - oldmillis) < loopLength) return false;
	oldmillis = millis();
	//Serial.println("Scanning beacons...");
	if (!isBTresponse("AT+DISI?", "OK+DISIS", 50))
	{
		setup();
		return false;
	}
	return true;
}

void BluetoothClient::jsonTable(String* json){
	StaticJsonBuffer<2000> jsonBuffer;
	JsonObject& root = jsonBuffer.createObject();
	JsonArray& beacons = root.createNestedArray("beacons");

	int k = 0;
	while((BeaconTable[k].lastActive != 0&&(k < MAX_BEACONS)))
	{
		String BeaconData = String(BeaconTable[k].data);
		JsonObject& beacon = beacons.createNestedObject();
		beacon["id"] = BeaconData.substring(0,8);
		beacon["uuid"] = BeaconData.substring(9,41);
		beacon["major"] = BeaconData.substring(42,46);
		beacon["minor"] = BeaconData.substring(46,50);
		beacon["power"] = BeaconData.substring(50,52);
		beacon["mac"] = BeaconData.substring(53,65);
		beacon["rssi"] = BeaconData.substring(66);
		beacon["lastactive"] = String((millis() - BeaconTable[k].lastActive)/1000);
		beacon["ticks"] = String(BeaconTable[k].advertisementsCnt);
		k++;
	}
	root.prettyPrintTo(*json); //just print after debug!!!
}


bool BluetoothClient::readBytesBT(char *buff, int bytesNo, long timeout){ //adds 0 at the end, strictly ascii communication
	btSer.setTimeout(timeout);
	int bytesRed = btSer.readBytes(buff, bytesNo);
	return bytesRed == bytesNo;
}

void BluetoothClient::writeBytesBT(char const *buff){ //ascii comms, 0 terminated
	btSer.print(buff);
}

bool BluetoothClient::isBTresponse(char const *sendbuff, char const *buff, long timeout){
	btSer.setTimeout(timeout);
	btSer.print(sendbuff);
	return btSer.find(buff);
}
